document.addEventListener("DOMContentLoaded", () => {
  initializeAdmin()
})

let currentSection = "dashboard"

function initializeAdmin() {
  // Initialize navigation
  initializeNavigation()

  // Load initial data
  loadDashboardData()

  // Initialize search and filters
  initializeControls()
}

function initializeNavigation() {
  const navLinks = document.querySelectorAll(".nav-link")

  navLinks.forEach((link) => {
    link.addEventListener("click", (e) => {
      e.preventDefault()
      const section = link.dataset.section

      // Update active nav
      navLinks.forEach((l) => l.classList.remove("active"))
      link.classList.add("active")

      // Show section
      showSection(section)
    })
  })

  // Initialize modals
  initializeModals()
}

function showSection(section) {
  // Hide all sections
  document.querySelectorAll(".admin-section").forEach((s) => s.classList.remove("active"))

  // Show target section
  document.getElementById(section).classList.add("active")
  currentSection = section

  // Load section data
  switch (section) {
    case "dashboard":
      loadDashboardData()
      break
    case "users":
      loadUsers()
      break
    case "posts":
      loadPosts()
      break
    case "tickets":
      loadTickets()
      break
    case "verification":
      loadVerificationRequests()
      break
  }
}

function initializeControls() {
  // User search
  const userSearch = document.getElementById("userSearch")
  const userFilter = document.getElementById("userFilter")

  if (userSearch) {
    let searchTimeout
    userSearch.addEventListener("input", () => {
      clearTimeout(searchTimeout)
      searchTimeout = setTimeout(() => {
        if (currentSection === "users") loadUsers()
      }, 300)
    })
  }

  if (userFilter) {
    userFilter.addEventListener("change", () => {
      if (currentSection === "users") loadUsers()
    })
  }

  // Post search
  const postSearch = document.getElementById("postSearch")
  const postFilter = document.getElementById("postFilter")

  if (postSearch) {
    let searchTimeout
    postSearch.addEventListener("input", () => {
      clearTimeout(searchTimeout)
      searchTimeout = setTimeout(() => {
        if (currentSection === "posts") loadPosts()
      }, 300)
    })
  }

  if (postFilter) {
    postFilter.addEventListener("change", () => {
      if (currentSection === "posts") loadPosts()
    })
  }

  // Ticket filter
  const ticketFilter = document.getElementById("ticketFilter")
  if (ticketFilter) {
    ticketFilter.addEventListener("change", () => {
      if (currentSection === "tickets") loadTickets()
    })
  }

  // WhatsApp form
  const whatsappForm = document.getElementById("whatsappForm")
  if (whatsappForm) {
    whatsappForm.addEventListener("submit", handleWhatsAppUpdate)
  }
}

function loadDashboardData() {
  const recentActivity = document.getElementById("recentActivity")
  recentActivity.innerHTML = "Loading recent activity..."

  // This would typically load recent user registrations, posts, etc.
  setTimeout(() => {
    recentActivity.innerHTML = `
      <div class="activity-item">
        <p>System is running smoothly</p>
        <small>All services operational</small>
      </div>
    `
  }, 1000)
}

function loadUsers(page = 1) {
  const container = document.getElementById("usersTable")
  const search = document.getElementById("userSearch")?.value || ""
  const filter = document.getElementById("userFilter")?.value || "all"

  container.innerHTML = '<div class="loading">Loading users...</div>'

  fetch(`admin/api/get-users.php?search=${encodeURIComponent(search)}&filter=${filter}&page=${page}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        if (data.users.length === 0) {
          container.innerHTML = '<div class="no-data">No users found</div>'
          return
        }

        let html = '<div class="data-table">'
        html += '<div class="table-header">Users Management</div>'

        data.users.forEach((user) => {
          html += `
            <div class="table-row">
              <div class="row-info">
                <img src="uploads/profiles/${user.profile_picture}" alt="Profile" class="row-avatar">
                <div class="row-details">
                  <h4>${user.full_name} ${user.is_verified ? '<span class="status-badge verified">Verified</span>' : ""} ${user.is_banned ? '<span class="status-badge banned">Banned</span>' : ""}</h4>
                  <p>@${user.username} • ${user.email}</p>
                  <p>${user.posts_count} posts • ${user.followers_count} followers</p>
                </div>
              </div>
              <div class="row-actions">
                <button class="action-btn primary" onclick="viewUser(${user.id})">View</button>
                ${user.is_banned ? `<button class="action-btn secondary" onclick="manageUser(${user.id}, 'unban')">Unban</button>` : `<button class="action-btn danger" onclick="manageUser(${user.id}, 'ban')">Ban</button>`}
                <button class="action-btn danger" onclick="manageUser(${user.id}, 'delete_posts')">Delete Posts</button>
              </div>
            </div>
          `
        })

        html += "</div>"
        container.innerHTML = html
      } else {
        container.innerHTML = '<div class="no-data">Failed to load users</div>'
      }
    })
    .catch((error) => {
      console.error("Error loading users:", error)
      container.innerHTML = '<div class="no-data">Error loading users</div>'
    })
}

function loadPosts(page = 1) {
  const container = document.getElementById("postsTable")
  const search = document.getElementById("postSearch")?.value || ""
  const filter = document.getElementById("postFilter")?.value || "all"

  container.innerHTML = '<div class="loading">Loading posts...</div>'

  fetch(`admin/api/get-posts.php?search=${encodeURIComponent(search)}&filter=${filter}&page=${page}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        if (data.posts.length === 0) {
          container.innerHTML = '<div class="no-data">No posts found</div>'
          return
        }

        let html = '<div class="data-table">'
        html += '<div class="table-header">Posts Management</div>'

        data.posts.forEach((post) => {
          html += `
            <div class="table-row">
              <div class="row-info">
                <img src="${post.media_url}" alt="Post" class="row-avatar" style="border-radius: 8px;">
                <div class="row-details">
                  <h4>By @${post.username} ${post.is_featured ? '<span class="status-badge featured">Featured</span>' : ""}</h4>
                  <p>${post.content ? post.content.substring(0, 50) + "..." : "No caption"}</p>
                  <p>${post.media_type} • ${post.likes_count} likes • ${post.comments_count} comments</p>
                </div>
              </div>
              <div class="row-actions">
                ${post.is_featured ? `<button class="action-btn secondary" onclick="managePost(${post.id}, 'unfeature')">Unfeature</button>` : `<button class="action-btn primary" onclick="managePost(${post.id}, 'feature')">Feature</button>`}
                <button class="action-btn danger" onclick="managePost(${post.id}, 'delete')">Delete</button>
              </div>
            </div>
          `
        })

        html += "</div>"
        container.innerHTML = html
      } else {
        container.innerHTML = '<div class="no-data">Failed to load posts</div>'
      }
    })
    .catch((error) => {
      console.error("Error loading posts:", error)
      container.innerHTML = '<div class="no-data">Error loading posts</div>'
    })
}

function loadTickets() {
  const container = document.getElementById("ticketsTable")
  const filter = document.getElementById("ticketFilter")?.value || "all"

  container.innerHTML = '<div class="loading">Loading tickets...</div>'

  fetch(`admin/api/get-tickets.php?filter=${filter}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        if (data.tickets.length === 0) {
          container.innerHTML = '<div class="no-data">No tickets found</div>'
          return
        }

        let html = '<div class="data-table">'
        html += '<div class="table-header">Support Tickets</div>'

        data.tickets.forEach((ticket) => {
          html += `
            <div class="table-row">
              <div class="row-info">
                <img src="uploads/profiles/${ticket.profile_picture}" alt="Profile" class="row-avatar">
                <div class="row-details">
                  <h4>${ticket.subject} <span class="status-badge ${ticket.status}">${ticket.status}</span></h4>
                  <p>From: ${ticket.full_name} (@${ticket.username})</p>
                  <p>${ticket.message.substring(0, 100)}...</p>
                </div>
              </div>
              <div class="row-actions">
                <button class="action-btn primary" onclick="viewTicket(${ticket.id})">View & Reply</button>
              </div>
            </div>
          `
        })

        html += "</div>"
        container.innerHTML = html
      } else {
        container.innerHTML = '<div class="no-data">Failed to load tickets</div>'
      }
    })
    .catch((error) => {
      console.error("Error loading tickets:", error)
      container.innerHTML = '<div class="no-data">Error loading tickets</div>'
    })
}

function loadVerificationRequests() {
  const container = document.getElementById("verificationTable")
  container.innerHTML = '<div class="loading">Loading verification requests...</div>'

  fetch("admin/api/get-users.php?filter=verification_requests")
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        // Filter for verification requests
        const requests = data.users.filter((user) => user.verification_requested && !user.is_verified)

        if (requests.length === 0) {
          container.innerHTML = '<div class="no-data">No verification requests</div>'
          return
        }

        let html = '<div class="data-table">'
        html += '<div class="table-header">Verification Requests</div>'

        requests.forEach((user) => {
          html += `
            <div class="table-row">
              <div class="row-info">
                <img src="uploads/profiles/${user.profile_picture}" alt="Profile" class="row-avatar">
                <div class="row-details">
                  <h4>${user.full_name}</h4>
                  <p>@${user.username} • ${user.email}</p>
                  <p>${user.followers_count} followers • ${user.posts_count} posts</p>
                </div>
              </div>
              <div class="row-actions">
                <button class="action-btn primary" onclick="manageUser(${user.id}, 'verify')">Approve</button>
                <button class="action-btn danger" onclick="rejectVerification(${user.id})">Reject</button>
              </div>
            </div>
          `
        })

        html += "</div>"
        container.innerHTML = html
      } else {
        container.innerHTML = '<div class="no-data">Failed to load verification requests</div>'
      }
    })
    .catch((error) => {
      console.error("Error loading verification requests:", error)
      container.innerHTML = '<div class="no-data">Error loading verification requests</div>'
    })
}

function manageUser(userId, action) {
  if (!confirm(`Are you sure you want to ${action} this user?`)) return

  fetch("admin/api/manage-user.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      user_id: userId,
      action: action,
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        alert(data.message)
        if (currentSection === "users") loadUsers()
        if (currentSection === "verification") loadVerificationRequests()
      } else {
        alert(data.message || "Action failed")
      }
    })
    .catch((error) => {
      console.error("Error:", error)
      alert("Action failed")
    })
}

function managePost(postId, action) {
  if (!confirm(`Are you sure you want to ${action} this post?`)) return

  fetch("admin/api/manage-post.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      post_id: postId,
      action: action,
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        alert(data.message)
        loadPosts()
      } else {
        alert(data.message || "Action failed")
      }
    })
    .catch((error) => {
      console.error("Error:", error)
      alert("Action failed")
    })
}

function viewTicket(ticketId) {
  // This would open a modal with ticket details and reply form
  alert("Ticket view functionality would be implemented here")
}

function handleWhatsAppUpdate(e) {
  e.preventDefault()
  const whatsappLink = document.getElementById("whatsappLink").value

  fetch("admin/api/update-settings.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      whatsapp_link: whatsappLink,
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        alert(data.message)
      } else {
        alert(data.message || "Update failed")
      }
    })
    .catch((error) => {
      console.error("Error:", error)
      alert("Update failed")
    })
}

function initializeModals() {
  const modals = document.querySelectorAll(".modal")
  const closeButtons = document.querySelectorAll(".close-modal")

  closeButtons.forEach((btn) => {
    btn.addEventListener("click", () => {
      modals.forEach((modal) => modal.classList.remove("active"))
    })
  })

  modals.forEach((modal) => {
    modal.addEventListener("click", (e) => {
      if (e.target === modal) {
        modal.classList.remove("active")
      }
    })
  })
}
