document.addEventListener("DOMContentLoaded", () => {
  initializeChat()
})

let lastMessageId = 0
let messagePollingInterval
let receiverId

function initializeChat() {
  receiverId = document.getElementById("receiverId").value
  const chatMessages = document.getElementById("chatMessages")
  const chatForm = document.getElementById("chatForm")
  const messageInput = document.getElementById("messageInput")

  // Get last message ID
  const messages = chatMessages.querySelectorAll(".message")
  if (messages.length > 0) {
    const lastMessage = messages[messages.length - 1]
    lastMessageId = Number.parseInt(lastMessage.dataset.messageId) || 0
  }

  // Initialize form submission
  chatForm.addEventListener("submit", handleMessageSubmit)

  // Auto-resize textarea
  messageInput.addEventListener("input", function () {
    this.style.height = "auto"
    this.style.height = this.scrollHeight + "px"
  })

  // Handle Enter key (send message) and Shift+Enter (new line)
  messageInput.addEventListener("keydown", (e) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault()
      chatForm.dispatchEvent(new Event("submit"))
    }
  })

  // Scroll to bottom
  scrollToBottom()

  // Start polling for new messages
  startMessagePolling()

  // Update online status
  updateOnlineStatus()
  setInterval(updateOnlineStatus, 30000)
}

function handleMessageSubmit(e) {
  e.preventDefault()
  const messageInput = document.getElementById("messageInput")
  const sendBtn = document.getElementById("sendBtn")
  const message = messageInput.value.trim()

  if (!message) return

  // Disable form during sending
  sendBtn.disabled = true
  messageInput.disabled = true

  // Add message optimistically
  addMessageToChat(message, true, "Sending...")

  fetch("../api/send-message.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      receiver_id: receiverId,
      message: message,
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        // Update the optimistic message with real data
        const optimisticMessage = document.querySelector(".message.sending")
        if (optimisticMessage) {
          optimisticMessage.classList.remove("sending")
          optimisticMessage.dataset.messageId = data.message_id
          const timeSpan = optimisticMessage.querySelector(".message-time")
          timeSpan.textContent = data.timestamp
        }

        // Clear form
        messageInput.value = ""
        messageInput.style.height = "auto"
        scrollToBottom()
      } else {
        // Remove optimistic message on error
        const optimisticMessage = document.querySelector(".message.sending")
        if (optimisticMessage) {
          optimisticMessage.remove()
        }
        alert(data.message || "Failed to send message")
      }
    })
    .catch((error) => {
      console.error("Error sending message:", error)
      // Remove optimistic message on error
      const optimisticMessage = document.querySelector(".message.sending")
      if (optimisticMessage) {
        optimisticMessage.remove()
      }
      alert("Failed to send message. Please try again.")
    })
    .finally(() => {
      // Re-enable form
      sendBtn.disabled = false
      messageInput.disabled = false
      messageInput.focus()
    })
}

function addMessageToChat(message, isSent, timestamp = null) {
  const chatMessages = document.getElementById("chatMessages")
  const messageDiv = document.createElement("div")
  messageDiv.className = `message ${isSent ? "sent" : "received"} ${timestamp === "Sending..." ? "sending" : ""}`

  messageDiv.innerHTML = `
    <div class="message-content">
      <p>${message.replace(/\n/g, "<br>")}</p>
      <span class="message-time">${timestamp || new Date().toLocaleTimeString("en-US", { hour: "2-digit", minute: "2-digit", hour12: false })}</span>
    </div>
  `

  chatMessages.appendChild(messageDiv)
  scrollToBottom()
}

function startMessagePolling() {
  messagePollingInterval = setInterval(() => {
    pollForNewMessages()
  }, 2000) // Poll every 2 seconds
}

function pollForNewMessages() {
  fetch(`../api/get-messages.php?user_id=${receiverId}&last_id=${lastMessageId}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.success && data.messages.length > 0) {
        data.messages.forEach((message) => {
          if (!message.is_sent) {
            addMessageToChat(message.message, false, message.timestamp)
          }
          lastMessageId = Math.max(lastMessageId, message.id)
        })
      }
    })
    .catch((error) => {
      console.error("Error polling messages:", error)
    })
}

function scrollToBottom() {
  const chatMessages = document.getElementById("chatMessages")
  chatMessages.scrollTop = chatMessages.scrollHeight
}

function updateOnlineStatus() {
  const onlineStatus = document.querySelector(".online-status")
  const userId = onlineStatus.dataset.userId

  // Simplified online status - in production, this would check real user activity
  if (Math.random() > 0.5) {
    onlineStatus.textContent = "Online"
    onlineStatus.classList.add("online")
  } else {
    onlineStatus.textContent = "Last seen recently"
    onlineStatus.classList.remove("online")
  }
}

// Clean up polling when leaving page
window.addEventListener("beforeunload", () => {
  if (messagePollingInterval) {
    clearInterval(messagePollingInterval)
  }
})

// Handle page visibility changes
document.addEventListener("visibilitychange", () => {
  if (document.hidden) {
    // Stop polling when page is hidden
    if (messagePollingInterval) {
      clearInterval(messagePollingInterval)
    }
  } else {
    // Resume polling when page is visible
    startMessagePolling()
  }
})
