document.addEventListener("DOMContentLoaded", () => {
  const fileInput = document.getElementById("media")
  const fileUploadArea = document.getElementById("fileUploadArea")
  const filePreview = document.getElementById("filePreview")
  const imagePreview = document.getElementById("imagePreview")
  const videoPreview = document.getElementById("videoPreview")
  const fileName = document.getElementById("fileName")
  const removeFile = document.getElementById("removeFile")
  const uploadPlaceholder = fileUploadArea.querySelector(".upload-placeholder")
  const postForm = document.getElementById("postForm")
  const submitBtn = document.getElementById("submitBtn")
  const loadingSpinner = document.getElementById("loadingSpinner")
  const btnText = submitBtn.querySelector(".btn-text")

  // File input change handler
  fileInput.addEventListener("change", handleFileSelect)

  // Drag and drop handlers
  fileUploadArea.addEventListener("dragover", handleDragOver)
  fileUploadArea.addEventListener("dragleave", handleDragLeave)
  fileUploadArea.addEventListener("drop", handleDrop)

  // Remove file handler
  removeFile.addEventListener("click", clearFileSelection)

  // Form submit handler
  postForm.addEventListener("submit", handleFormSubmit)

  function handleFileSelect(e) {
    const file = e.target.files[0]
    if (file) {
      displayFilePreview(file)
    }
  }

  function handleDragOver(e) {
    e.preventDefault()
    fileUploadArea.classList.add("dragover")
  }

  function handleDragLeave(e) {
    e.preventDefault()
    fileUploadArea.classList.remove("dragover")
  }

  function handleDrop(e) {
    e.preventDefault()
    fileUploadArea.classList.remove("dragover")

    const files = e.dataTransfer.files
    if (files.length > 0) {
      fileInput.files = files
      displayFilePreview(files[0])
    }
  }

  function displayFilePreview(file) {
    const fileSize = (file.size / (1024 * 1024)).toFixed(2) // Size in MB
    const maxSize = 50 // 50MB limit

    // Check file size
    if (fileSize > maxSize) {
      alert(`File size (${fileSize}MB) exceeds the maximum limit of ${maxSize}MB`)
      clearFileSelection()
      return
    }

    // Check file type
    const allowedTypes = ["image/jpeg", "image/jpg", "image/png", "image/gif", "video/mp4", "video/avi", "video/mov"]
    if (!allowedTypes.includes(file.type)) {
      alert("Invalid file type. Please select an image (JPG, PNG, GIF) or video (MP4, AVI, MOV)")
      clearFileSelection()
      return
    }

    // Hide placeholder and show preview
    uploadPlaceholder.style.display = "none"
    filePreview.style.display = "block"

    // Set file name
    fileName.textContent = `${file.name} (${fileSize}MB)`

    // Create file URL for preview
    const fileURL = URL.createObjectURL(file)

    if (file.type.startsWith("image/")) {
      imagePreview.src = fileURL
      imagePreview.style.display = "block"
      videoPreview.style.display = "none"
    } else if (file.type.startsWith("video/")) {
      videoPreview.src = fileURL
      videoPreview.style.display = "block"
      imagePreview.style.display = "none"
    }
  }

  function clearFileSelection() {
    fileInput.value = ""
    uploadPlaceholder.style.display = "block"
    filePreview.style.display = "none"
    imagePreview.style.display = "none"
    videoPreview.style.display = "none"

    // Revoke object URLs to free memory
    if (imagePreview.src) {
      URL.revokeObjectURL(imagePreview.src)
      imagePreview.src = ""
    }
    if (videoPreview.src) {
      URL.revokeObjectURL(videoPreview.src)
      videoPreview.src = ""
    }
  }

  function handleFormSubmit(e) {
    // Show loading state
    submitBtn.disabled = true
    btnText.style.display = "none"
    loadingSpinner.style.display = "inline-block"

    // Form will submit normally, loading state will be visible during upload
  }

  // Validate form before submit
  function validateForm() {
    if (!fileInput.files || fileInput.files.length === 0) {
      alert("Please select a file to upload")
      return false
    }
    return true
  }

  // Add form validation
  postForm.addEventListener("submit", (e) => {
    if (!validateForm()) {
      e.preventDefault()
      submitBtn.disabled = false
      btnText.style.display = "inline"
      loadingSpinner.style.display = "none"
    }
  })
})
