document.addEventListener("DOMContentLoaded", () => {
  initializeInbox()
})

function initializeInbox() {
  // Initialize conversation clicks
  const conversationItems = document.querySelectorAll(".conversation-item")
  conversationItems.forEach((item) => {
    item.addEventListener("click", () => {
      const userId = item.dataset.userId
      window.location.href = `pages/chat.php?user=${userId}`
    })
  })

  // Initialize new message modal
  initializeNewMessageModal()

  // Update online status indicators
  updateOnlineStatus()
  setInterval(updateOnlineStatus, 30000) // Update every 30 seconds
}

function initializeNewMessageModal() {
  const newMessageBtn = document.getElementById("newMessageBtn")
  const modal = document.getElementById("newMessageModal")
  const closeBtn = document.getElementById("closeNewMessageModal")
  const searchInput = document.getElementById("userSearch")
  const searchResults = document.getElementById("searchResults")

  newMessageBtn.addEventListener("click", () => {
    modal.classList.add("active")
    searchInput.focus()
  })

  closeBtn.addEventListener("click", () => {
    modal.classList.remove("active")
    searchInput.value = ""
    searchResults.innerHTML = ""
  })

  modal.addEventListener("click", (e) => {
    if (e.target === modal) {
      modal.classList.remove("active")
      searchInput.value = ""
      searchResults.innerHTML = ""
    }
  })

  // Search functionality
  let searchTimeout
  searchInput.addEventListener("input", () => {
    clearTimeout(searchTimeout)
    const query = searchInput.value.trim()

    if (query.length < 2) {
      searchResults.innerHTML = ""
      return
    }

    searchTimeout = setTimeout(() => {
      searchUsers(query)
    }, 300)
  })
}

function searchUsers(query) {
  const searchResults = document.getElementById("searchResults")
  searchResults.innerHTML = '<div class="loading">Searching...</div>'

  fetch(`api/search-users.php?q=${encodeURIComponent(query)}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        if (data.users.length === 0) {
          searchResults.innerHTML = '<div class="no-results">No users found</div>'
        } else {
          searchResults.innerHTML = data.users
            .map(
              (user) => `
            <div class="user-result" data-user-id="${user.id}">
              <img src="uploads/profiles/${user.profile_picture}" alt="Profile">
              <div class="user-result-info">
                <h4>${user.full_name}${user.is_verified ? ' <span class="verified-badge">✓</span>' : ""}</h4>
                <p>@${user.username}</p>
              </div>
            </div>
          `,
            )
            .join("")

          // Add click handlers to user results
          searchResults.querySelectorAll(".user-result").forEach((result) => {
            result.addEventListener("click", () => {
              const userId = result.dataset.userId
              window.location.href = `pages/chat.php?user=${userId}`
            })
          })
        }
      } else {
        searchResults.innerHTML = '<div class="error">Search failed</div>'
      }
    })
    .catch((error) => {
      console.error("Search error:", error)
      searchResults.innerHTML = '<div class="error">Search failed</div>'
    })
}

function updateOnlineStatus() {
  // This is a simplified online status system
  // In a real application, you would track user activity and last seen timestamps
  const onlineIndicators = document.querySelectorAll(".online-indicator")

  onlineIndicators.forEach((indicator) => {
    // Randomly show some users as online for demo purposes
    // In production, this would check actual user activity
    if (Math.random() > 0.7) {
      indicator.classList.add("online")
    } else {
      indicator.classList.remove("online")
    }
  })
}
