// Main JavaScript for Show U Social Media App

document.addEventListener("DOMContentLoaded", () => {
  // Initialize like buttons
  initializeLikeButtons()

  // Initialize comment buttons
  initializeCommentButtons()

  // Initialize share buttons
  initializeShareButtons()
})

function initializeLikeButtons() {
  const likeButtons = document.querySelectorAll(".like-btn")

  likeButtons.forEach((button) => {
    button.addEventListener("click", function () {
      const postId = this.dataset.postId
      const likeCount = this.querySelector(".like-count")
      const isLiked = this.classList.contains("liked")

      // Optimistic UI update
      if (isLiked) {
        this.classList.remove("liked")
        likeCount.textContent = Number.parseInt(likeCount.textContent) - 1
      } else {
        this.classList.add("liked")
        likeCount.textContent = Number.parseInt(likeCount.textContent) + 1
      }

      // Send AJAX request
      fetch("api/toggle-like.php", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ post_id: postId }),
      })
        .then((response) => response.json())
        .then((data) => {
          if (!data.success) {
            // Revert optimistic update on error
            if (isLiked) {
              this.classList.add("liked")
              likeCount.textContent = Number.parseInt(likeCount.textContent) + 1
            } else {
              this.classList.remove("liked")
              likeCount.textContent = Number.parseInt(likeCount.textContent) - 1
            }
            alert("Failed to update like. Please try again.")
          }
        })
        .catch((error) => {
          console.error("Error:", error)
          // Revert optimistic update on error
          if (isLiked) {
            this.classList.add("liked")
            likeCount.textContent = Number.parseInt(likeCount.textContent) + 1
          } else {
            this.classList.remove("liked")
            likeCount.textContent = Number.parseInt(likeCount.textContent) - 1
          }
        })
    })
  })
}

function initializeCommentButtons() {
  const commentButtons = document.querySelectorAll(".comment-btn")

  commentButtons.forEach((button) => {
    button.addEventListener("click", function () {
      const postId = this.dataset.postId
      const postCard = this.closest(".post-card")
      let commentsSection = postCard.querySelector(".comments-section")

      if (commentsSection) {
        // Toggle comments visibility
        commentsSection.style.display = commentsSection.style.display === "none" ? "block" : "none"
      } else {
        // Create and load comments section
        commentsSection = createCommentsSection(postId)
        postCard.appendChild(commentsSection)
        loadComments(postId)
      }
    })
  })
}

function createCommentsSection(postId) {
  const section = document.createElement("div")
  section.className = "comments-section"
  section.innerHTML = `
    <div class="comments-list" id="comments-${postId}">
      <div class="loading-comments">Loading comments...</div>
    </div>
    <form class="comment-form" data-post-id="${postId}">
      <textarea class="comment-input" placeholder="Add a comment..." maxlength="500" rows="1"></textarea>
      <button type="submit" class="comment-submit">→</button>
    </form>
  `

  // Add form submit handler
  const form = section.querySelector(".comment-form")
  form.addEventListener("submit", handleCommentSubmit)

  // Auto-resize textarea
  const textarea = section.querySelector(".comment-input")
  textarea.addEventListener("input", function () {
    this.style.height = "auto"
    this.style.height = this.scrollHeight + "px"
  })

  return section
}

function loadComments(postId) {
  fetch(`api/get-comments.php?post_id=${postId}`)
    .then((response) => response.json())
    .then((data) => {
      const commentsList = document.getElementById(`comments-${postId}`)
      if (data.success) {
        if (data.comments.length === 0) {
          commentsList.innerHTML = '<p class="no-comments">No comments yet. Be the first to comment!</p>'
        } else {
          commentsList.innerHTML = data.comments
            .map(
              (comment) => `
            <div class="comment-item">
              <img src="uploads/profiles/${comment.profile_picture}" alt="Profile" class="comment-avatar">
              <div class="comment-content">
                <div class="comment-header">
                  <span class="comment-author">${comment.full_name}${comment.is_verified ? ' <span class="verified-badge">✓</span>' : ""}</span>
                  <span class="comment-time">${comment.time_ago}</span>
                </div>
                <p class="comment-text">${comment.comment}</p>
              </div>
            </div>
          `,
            )
            .join("")
        }
      } else {
        commentsList.innerHTML = '<p class="error">Failed to load comments.</p>'
      }
    })
    .catch((error) => {
      console.error("Error loading comments:", error)
      const commentsList = document.getElementById(`comments-${postId}`)
      commentsList.innerHTML = '<p class="error">Failed to load comments.</p>'
    })
}

function handleCommentSubmit(e) {
  e.preventDefault()
  const form = e.target
  const postId = form.dataset.postId
  const textarea = form.querySelector(".comment-input")
  const submitBtn = form.querySelector(".comment-submit")
  const comment = textarea.value.trim()

  if (!comment) return

  // Disable form during submission
  submitBtn.disabled = true
  textarea.disabled = true

  fetch("api/add-comment.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      post_id: postId,
      comment: comment,
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        // Add new comment to the list
        const commentsList = document.getElementById(`comments-${postId}`)
        const noComments = commentsList.querySelector(".no-comments")
        if (noComments) {
          commentsList.innerHTML = ""
        }

        const newComment = document.createElement("div")
        newComment.className = "comment-item"
        newComment.innerHTML = `
          <img src="uploads/profiles/${data.comment.profile_picture}" alt="Profile" class="comment-avatar">
          <div class="comment-content">
            <div class="comment-header">
              <span class="comment-author">${data.comment.full_name}</span>
              <span class="comment-time">just now</span>
            </div>
            <p class="comment-text">${data.comment.comment}</p>
          </div>
        `
        commentsList.appendChild(newComment)

        // Update comment count
        const commentBtn = document.querySelector(`[data-post-id="${postId}"].comment-btn`)
        const commentCount = commentBtn.querySelector(".comment-count")
        commentCount.textContent = Number.parseInt(commentCount.textContent) + 1

        // Clear form
        textarea.value = ""
        textarea.style.height = "auto"
      } else {
        alert(data.message || "Failed to add comment")
      }
    })
    .catch((error) => {
      console.error("Error adding comment:", error)
      alert("Failed to add comment. Please try again.")
    })
    .finally(() => {
      // Re-enable form
      submitBtn.disabled = false
      textarea.disabled = false
    })
}

function initializeShareButtons() {
  const shareButtons = document.querySelectorAll(".share-btn")

  shareButtons.forEach((button) => {
    button.addEventListener("click", function () {
      const postId = this.dataset.postId
      const shareUrl = `${window.location.origin}/post.php?id=${postId}`

      if (navigator.share) {
        navigator.share({
          title: "Check out this post on Show U",
          url: shareUrl,
        })
      } else {
        // Fallback: copy to clipboard
        navigator.clipboard
          .writeText(shareUrl)
          .then(() => {
            alert("Link copied to clipboard!")
          })
          .catch(() => {
            // Fallback for older browsers
            const textArea = document.createElement("textarea")
            textArea.value = shareUrl
            document.body.appendChild(textArea)
            textArea.select()
            document.execCommand("copy")
            document.body.removeChild(textArea)
            alert("Link copied to clipboard!")
          })
      }
    })
  })
}

// Utility function for AJAX requests
function makeRequest(url, method = "GET", data = null) {
  const options = {
    method: method,
    headers: {
      "Content-Type": "application/json",
    },
  }

  if (data) {
    options.body = JSON.stringify(data)
  }

  return fetch(url, options)
    .then((response) => response.json())
    .catch((error) => {
      console.error("Request failed:", error)
      throw error
    })
}
