document.addEventListener("DOMContentLoaded", () => {
  initializeShorts()
})

let currentVideoIndex = 0
let videos = []
let isScrolling = false
let currentPostId = null

function initializeShorts() {
  const shortsContainer = document.getElementById("shortsContainer")
  videos = Array.from(document.querySelectorAll(".short-video"))

  if (videos.length === 0) return

  // Initialize video players
  videos.forEach((videoElement, index) => {
    const video = videoElement.querySelector(".video-player")
    const playPauseBtn = videoElement.querySelector(".play-pause-btn")
    const progressFill = videoElement.querySelector(".progress-fill")

    // Video click to play/pause
    video.addEventListener("click", () => togglePlayPause(videoElement))
    playPauseBtn.addEventListener("click", () => togglePlayPause(videoElement))

    // Update progress bar
    video.addEventListener("timeupdate", () => {
      if (video.duration) {
        const progress = (video.currentTime / video.duration) * 100
        progressFill.style.width = `${progress}%`
      }
    })

    // Loop video
    video.addEventListener("ended", () => {
      video.currentTime = 0
      video.play()
    })

    // Handle video load errors
    video.addEventListener("error", () => {
      console.error("Error loading video:", video.src)
      videoElement.style.display = "none"
    })
  })

  // Initialize scroll handling
  initializeScrollHandling()

  // Initialize interactions
  initializeShortsInteractions()

  // Initialize comments modal
  initializeCommentsModal()

  // Play first video
  if (videos[0]) {
    playVideo(0)
  }
}

function initializeScrollHandling() {
  const shortsContainer = document.getElementById("shortsContainer")
  let scrollTimeout

  shortsContainer.addEventListener("scroll", () => {
    if (isScrolling) return

    clearTimeout(scrollTimeout)
    scrollTimeout = setTimeout(() => {
      const containerHeight = shortsContainer.clientHeight
      const scrollTop = shortsContainer.scrollTop

      // Find which video is currently in view
      const newIndex = Math.round(scrollTop / containerHeight)

      if (newIndex !== currentVideoIndex && newIndex >= 0 && newIndex < videos.length) {
        pauseVideo(currentVideoIndex)
        currentVideoIndex = newIndex
        playVideo(currentVideoIndex)
      }
    }, 100)
  })

  // Touch/swipe handling for mobile
  let startY = 0
  let startTime = 0

  shortsContainer.addEventListener("touchstart", (e) => {
    startY = e.touches[0].clientY
    startTime = Date.now()
  })

  shortsContainer.addEventListener("touchend", (e) => {
    const endY = e.changedTouches[0].clientY
    const endTime = Date.now()
    const deltaY = startY - endY
    const deltaTime = endTime - startTime

    // Detect swipe (minimum distance and maximum time)
    if (Math.abs(deltaY) > 50 && deltaTime < 300) {
      if (deltaY > 0) {
        // Swipe up - next video
        scrollToVideo(currentVideoIndex + 1)
      } else {
        // Swipe down - previous video
        scrollToVideo(currentVideoIndex - 1)
      }
    }
  })

  // Keyboard navigation
  document.addEventListener("keydown", (e) => {
    if (e.key === "ArrowUp") {
      e.preventDefault()
      scrollToVideo(currentVideoIndex - 1)
    } else if (e.key === "ArrowDown") {
      e.preventDefault()
      scrollToVideo(currentVideoIndex + 1)
    } else if (e.key === " ") {
      e.preventDefault()
      togglePlayPause(videos[currentVideoIndex])
    }
  })
}

function scrollToVideo(index) {
  if (index < 0 || index >= videos.length) return

  isScrolling = true
  const shortsContainer = document.getElementById("shortsContainer")
  const targetScrollTop = index * shortsContainer.clientHeight

  shortsContainer.scrollTo({
    top: targetScrollTop,
    behavior: "smooth",
  })

  setTimeout(() => {
    isScrolling = false
  }, 500)
}

function playVideo(index) {
  if (index < 0 || index >= videos.length) return

  const videoElement = videos[index]
  const video = videoElement.querySelector(".video-player")

  // Pause all other videos
  videos.forEach((v, i) => {
    if (i !== index) {
      pauseVideo(i)
    }
  })

  // Play current video
  video.play().catch((error) => {
    console.error("Error playing video:", error)
  })

  videoElement.classList.remove("paused")

  // Preload next videos
  preloadAdjacentVideos(index)
}

function pauseVideo(index) {
  if (index < 0 || index >= videos.length) return

  const videoElement = videos[index]
  const video = videoElement.querySelector(".video-player")

  video.pause()
  videoElement.classList.add("paused")
}

function togglePlayPause(videoElement) {
  const video = videoElement.querySelector(".video-player")

  if (video.paused) {
    video.play()
    videoElement.classList.remove("paused")
  } else {
    video.pause()
    videoElement.classList.add("paused")
  }
}

function preloadAdjacentVideos(currentIndex) {
  // Preload next 2 and previous 1 video
  const indicesToPreload = [currentIndex - 1, currentIndex + 1, currentIndex + 2]

  indicesToPreload.forEach((index) => {
    if (index >= 0 && index < videos.length) {
      const video = videos[index].querySelector(".video-player")
      if (video.readyState < 2) {
        // HTMLMediaElement.HAVE_CURRENT_DATA
        video.load()
      }
    }
  })
}

function initializeShortsInteractions() {
  // Like buttons
  document.querySelectorAll(".like-btn").forEach((btn) => {
    btn.addEventListener("click", handleLikeClick)
  })

  // Comment buttons
  document.querySelectorAll(".comment-btn").forEach((btn) => {
    btn.addEventListener("click", handleCommentClick)
  })

  // Share buttons
  document.querySelectorAll(".share-btn").forEach((btn) => {
    btn.addEventListener("click", handleShareClick)
  })

  // Follow buttons
  document.querySelectorAll(".follow-btn-small").forEach((btn) => {
    btn.addEventListener("click", handleFollowClick)
  })
}

function handleLikeClick(e) {
  e.stopPropagation()
  const button = e.currentTarget
  const postId = button.dataset.postId
  const likeCount = button.querySelector(".action-count")
  const isLiked = button.classList.contains("liked")

  // Optimistic UI update
  if (isLiked) {
    button.classList.remove("liked")
    likeCount.textContent = Number.parseInt(likeCount.textContent) - 1
  } else {
    button.classList.add("liked")
    likeCount.textContent = Number.parseInt(likeCount.textContent) + 1
  }

  // Send AJAX request
  fetch("api/toggle-like.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({ post_id: postId }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (!data.success) {
        // Revert on error
        if (isLiked) {
          button.classList.add("liked")
          likeCount.textContent = Number.parseInt(likeCount.textContent) + 1
        } else {
          button.classList.remove("liked")
          likeCount.textContent = Number.parseInt(likeCount.textContent) - 1
        }
      }
    })
    .catch((error) => {
      console.error("Error:", error)
      // Revert on error
      if (isLiked) {
        button.classList.add("liked")
        likeCount.textContent = Number.parseInt(likeCount.textContent) + 1
      } else {
        button.classList.remove("liked")
        likeCount.textContent = Number.parseInt(likeCount.textContent) - 1
      }
    })
}

function handleCommentClick(e) {
  e.stopPropagation()
  const button = e.currentTarget
  const postId = button.dataset.postId
  openCommentsModal(postId)
}

function handleShareClick(e) {
  e.stopPropagation()
  const button = e.currentTarget
  const postId = button.dataset.postId
  const shareUrl = `${window.location.origin}/post.php?id=${postId}`

  if (navigator.share) {
    navigator.share({
      title: "Check out this video on Show U",
      url: shareUrl,
    })
  } else {
    navigator.clipboard.writeText(shareUrl).then(() => {
      // Show temporary feedback
      const originalIcon = button.querySelector(".action-icon").textContent
      button.querySelector(".action-icon").textContent = "✓"
      setTimeout(() => {
        button.querySelector(".action-icon").textContent = originalIcon
      }, 1000)
    })
  }
}

function handleFollowClick(e) {
  e.stopPropagation()
  const button = e.currentTarget
  const userId = button.dataset.userId
  const isFollowing = button.classList.contains("following")

  // Optimistic UI update
  if (isFollowing) {
    button.classList.remove("following")
    button.textContent = "Follow"
  } else {
    button.classList.add("following")
    button.textContent = "Following"
  }

  button.disabled = true

  fetch("api/toggle-follow.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({ user_id: userId }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (!data.success) {
        // Revert on error
        if (isFollowing) {
          button.classList.add("following")
          button.textContent = "Following"
        } else {
          button.classList.remove("following")
          button.textContent = "Follow"
        }
      }
    })
    .catch((error) => {
      console.error("Error:", error)
      // Revert on error
      if (isFollowing) {
        button.classList.add("following")
        button.textContent = "Following"
      } else {
        button.classList.remove("following")
        button.textContent = "Follow"
      }
    })
    .finally(() => {
      button.disabled = false
    })
}

function initializeCommentsModal() {
  const modal = document.getElementById("commentsModal")
  const closeBtn = document.getElementById("closeModal")
  const commentForm = document.getElementById("modalCommentForm")

  closeBtn.addEventListener("click", closeCommentsModal)
  modal.addEventListener("click", (e) => {
    if (e.target === modal) {
      closeCommentsModal()
    }
  })

  commentForm.addEventListener("submit", handleModalCommentSubmit)

  // Auto-resize textarea
  const textarea = commentForm.querySelector(".comment-input")
  textarea.addEventListener("input", function () {
    this.style.height = "auto"
    this.style.height = this.scrollHeight + "px"
  })
}

function openCommentsModal(postId) {
  currentPostId = postId
  const modal = document.getElementById("commentsModal")
  const commentsList = document.getElementById("modalCommentsList")

  modal.classList.add("active")
  commentsList.innerHTML = '<div class="loading-comments">Loading comments...</div>'

  // Pause current video
  if (videos[currentVideoIndex]) {
    pauseVideo(currentVideoIndex)
  }

  // Load comments
  fetch(`api/get-comments.php?post_id=${postId}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        if (data.comments.length === 0) {
          commentsList.innerHTML = '<p class="no-comments">No comments yet. Be the first to comment!</p>'
        } else {
          commentsList.innerHTML = data.comments
            .map(
              (comment) => `
            <div class="comment-item">
              <img src="uploads/profiles/${comment.profile_picture}" alt="Profile" class="comment-avatar">
              <div class="comment-content">
                <div class="comment-header">
                  <span class="comment-author">${comment.full_name}${comment.is_verified ? ' <span class="verified-badge">✓</span>' : ""}</span>
                  <span class="comment-time">${comment.time_ago}</span>
                </div>
                <p class="comment-text">${comment.comment}</p>
              </div>
            </div>
          `,
            )
            .join("")
        }
      } else {
        commentsList.innerHTML = '<p class="error">Failed to load comments.</p>'
      }
    })
    .catch((error) => {
      console.error("Error loading comments:", error)
      commentsList.innerHTML = '<p class="error">Failed to load comments.</p>'
    })
}

function closeCommentsModal() {
  const modal = document.getElementById("commentsModal")
  modal.classList.remove("active")
  currentPostId = null

  // Resume current video
  if (videos[currentVideoIndex]) {
    playVideo(currentVideoIndex)
  }
}

function handleModalCommentSubmit(e) {
  e.preventDefault()
  if (!currentPostId) return

  const form = e.target
  const textarea = form.querySelector(".comment-input")
  const submitBtn = form.querySelector(".comment-submit")
  const comment = textarea.value.trim()

  if (!comment) return

  submitBtn.disabled = true
  textarea.disabled = true

  fetch("api/add-comment.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      post_id: currentPostId,
      comment: comment,
    }),
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        // Add new comment to modal
        const commentsList = document.getElementById("modalCommentsList")
        const noComments = commentsList.querySelector(".no-comments")
        if (noComments) {
          commentsList.innerHTML = ""
        }

        const newComment = document.createElement("div")
        newComment.className = "comment-item"
        newComment.innerHTML = `
          <img src="uploads/profiles/${data.comment.profile_picture}" alt="Profile" class="comment-avatar">
          <div class="comment-content">
            <div class="comment-header">
              <span class="comment-author">${data.comment.full_name}</span>
              <span class="comment-time">just now</span>
            </div>
            <p class="comment-text">${data.comment.comment}</p>
          </div>
        `
        commentsList.appendChild(newComment)

        // Update comment count in video
        const commentBtn = document.querySelector(`[data-post-id="${currentPostId}"].comment-btn`)
        if (commentBtn) {
          const commentCount = commentBtn.querySelector(".action-count")
          commentCount.textContent = Number.parseInt(commentCount.textContent) + 1
        }

        // Clear form
        textarea.value = ""
        textarea.style.height = "auto"
      } else {
        alert(data.message || "Failed to add comment")
      }
    })
    .catch((error) => {
      console.error("Error adding comment:", error)
      alert("Failed to add comment. Please try again.")
    })
    .finally(() => {
      submitBtn.disabled = false
      textarea.disabled = false
    })
}

// Handle page visibility changes
document.addEventListener("visibilitychange", () => {
  if (document.hidden) {
    // Pause video when page is hidden
    if (videos[currentVideoIndex]) {
      pauseVideo(currentVideoIndex)
    }
  } else {
    // Resume video when page is visible
    if (videos[currentVideoIndex]) {
      playVideo(currentVideoIndex)
    }
  }
})
