<?php
require_once '../includes/config.php';
requireLogin();

$user = getCurrentUser();
$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $content = sanitizeInput($_POST['content']);
    $media_file = $_FILES['media'];
    
    // Validate file upload
    if (empty($media_file['name'])) {
        $error = 'Please select an image or video to upload';
    } elseif ($media_file['error'] !== UPLOAD_ERR_OK) {
        $error = 'File upload failed. Please try again.';
    } elseif ($media_file['size'] > MAX_FILE_SIZE) {
        $error = 'File size too large. Maximum size is 50MB.';
    } else {
        $file_extension = strtolower(pathinfo($media_file['name'], PATHINFO_EXTENSION));
        $media_type = '';
        
        // Determine media type
        if (in_array($file_extension, ALLOWED_IMAGE_TYPES)) {
            $media_type = 'image';
        } elseif (in_array($file_extension, ALLOWED_VIDEO_TYPES)) {
            $media_type = 'video';
        } else {
            $error = 'Invalid file type. Only images (jpg, jpeg, png, gif) and videos (mp4, avi, mov) are allowed.';
        }
        
        if (!$error) {
            // Create upload directory if it doesn't exist
            $upload_dir = '../uploads/posts/';
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            // Generate unique filename
            $filename = uniqid() . '_' . time() . '.' . $file_extension;
            $upload_path = $upload_dir . $filename;
            $media_url = 'uploads/posts/' . $filename;
            
            // Move uploaded file
            if (move_uploaded_file($media_file['tmp_name'], $upload_path)) {
                // Save post to database
                $pdo = getDBConnection();
                $stmt = $pdo->prepare("INSERT INTO posts (user_id, content, media_type, media_url) VALUES (?, ?, ?, ?)");
                
                if ($stmt->execute([$user['id'], $content, $media_type, $media_url])) {
                    // Update user's post count
                    $update_stmt = $pdo->prepare("UPDATE users SET posts_count = posts_count + 1 WHERE id = ?");
                    $update_stmt->execute([$user['id']]);
                    
                    $success = 'Post created successfully!';
                    
                    // Redirect to dashboard after 2 seconds
                    header("refresh:2;url=../dashboard.php");
                } else {
                    $error = 'Failed to save post. Please try again.';
                    // Delete uploaded file if database save failed
                    unlink($upload_path);
                }
            } else {
                $error = 'Failed to upload file. Please try again.';
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Post - Show U</title>
    <link rel="stylesheet" href="../assets/css/style.css">
</head>
<body>
    <div class="main-container">
        <header class="header">
            <div class="header-content">
                <a href="../dashboard.php" class="back-btn">←</a>
                <h1>Create Post</h1>
            </div>
        </header>
        
        <div class="content">
            <?php if ($error): ?>
                <div class="alert alert-error"><?php echo $error; ?></div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success"><?php echo $success; ?></div>
            <?php endif; ?>
            
            <form method="POST" enctype="multipart/form-data" class="create-post-form" id="postForm">
                <div class="form-group">
                    <label for="content">Caption (Optional)</label>
                    <textarea id="content" name="content" rows="4" placeholder="What's on your mind?"><?php echo isset($_POST['content']) ? htmlspecialchars($_POST['content']) : ''; ?></textarea>
                </div>
                
                <div class="form-group">
                    <label for="media">Upload Image or Video *</label>
                    <div class="file-upload-area" id="fileUploadArea">
                        <input type="file" id="media" name="media" accept="image/*,video/*" required>
                        <div class="upload-placeholder">
                            <div class="upload-icon">📷</div>
                            <p>Tap to select image or video</p>
                            <small>Max size: 50MB | Images: JPG, PNG, GIF | Videos: MP4, AVI, MOV</small>
                        </div>
                        <div class="file-preview" id="filePreview" style="display: none;">
                            <img id="imagePreview" style="display: none;">
                            <video id="videoPreview" controls style="display: none;"></video>
                            <div class="file-info">
                                <span id="fileName"></span>
                                <button type="button" class="remove-file" id="removeFile">×</button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="window.location.href='../dashboard.php'">Cancel</button>
                    <button type="submit" class="btn btn-primary" id="submitBtn">
                        <span class="btn-text">Share Post</span>
                        <span class="loading" id="loadingSpinner" style="display: none;"></span>
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <script src="../assets/js/create-post.js"></script>
</body>
</html>
