<?php
require_once 'includes/config.php';
requireLogin();

$current_user = getCurrentUser();
$profile_user_id = isset($_GET['id']) ? (int)$_GET['id'] : $current_user['id'];
$is_own_profile = ($profile_user_id == $current_user['id']);

$pdo = getDBConnection();

// Get profile user data
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND is_banned = 0");
$stmt->execute([$profile_user_id]);
$profile_user = $stmt->fetch();

if (!$profile_user) {
    header('Location: dashboard.php');
    exit();
}

// Check if current user follows this profile
$is_following = false;
if (!$is_own_profile) {
    $stmt = $pdo->prepare("SELECT id FROM follows WHERE follower_id = ? AND following_id = ?");
    $stmt->execute([$current_user['id'], $profile_user_id]);
    $is_following = $stmt->fetch() ? true : false;
}

// Get user's posts
$stmt = $pdo->prepare("
    SELECT p.*, 
           (SELECT COUNT(*) FROM likes WHERE post_id = p.id) as likes_count,
           (SELECT COUNT(*) FROM comments WHERE post_id = p.id) as comments_count,
           (SELECT COUNT(*) FROM likes WHERE post_id = p.id AND user_id = ?) as user_liked
    FROM posts p 
    WHERE p.user_id = ? 
    ORDER BY p.created_at DESC
");
$stmt->execute([$current_user['id'], $profile_user_id]);
$posts = $stmt->fetchAll();

// Separate images and videos
$images = array_filter($posts, function($post) { return $post['media_type'] == 'image'; });
$videos = array_filter($posts, function($post) { return $post['media_type'] == 'video'; });
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($profile_user['full_name']); ?> - Show U</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="main-container">
        <header class="header">
            <div class="header-content">
                <?php if (!$is_own_profile): ?>
                    <a href="dashboard.php" class="back-btn">←</a>
                <?php endif; ?>
                <h1><?php echo $is_own_profile ? 'My Profile' : htmlspecialchars($profile_user['username']); ?></h1>
                <?php if ($is_own_profile): ?>
                    <a href="settings.php" class="settings-btn">⚙</a>
                <?php endif; ?>
            </div>
        </header>
        
        <div class="content">
            <div class="profile-header">
                <div class="profile-avatar">
                    <img src="uploads/profiles/<?php echo htmlspecialchars($profile_user['profile_picture']); ?>" 
                         alt="Profile Picture" class="profile-picture-large">
                </div>
                
                <div class="profile-info">
                    <h2><?php echo htmlspecialchars($profile_user['full_name']); ?>
                        <?php if ($profile_user['is_verified']): ?>
                            <span class="verified-badge">✓</span>
                        <?php endif; ?>
                    </h2>
                    <p class="username">@<?php echo htmlspecialchars($profile_user['username']); ?></p>
                    
                    <?php if ($profile_user['bio']): ?>
                        <p class="bio"><?php echo nl2br(htmlspecialchars($profile_user['bio'])); ?></p>
                    <?php endif; ?>
                    
                    <div class="profile-stats">
                        <div class="stat">
                            <span class="stat-number"><?php echo $profile_user['posts_count']; ?></span>
                            <span class="stat-label">Posts</span>
                        </div>
                        <div class="stat">
                            <span class="stat-number"><?php echo $profile_user['followers_count']; ?></span>
                            <span class="stat-label">Followers</span>
                        </div>
                        <div class="stat">
                            <span class="stat-number"><?php echo $profile_user['following_count']; ?></span>
                            <span class="stat-label">Following</span>
                        </div>
                    </div>
                    
                    <?php if (!$is_own_profile): ?>
                        <button class="btn follow-btn <?php echo $is_following ? 'following' : 'btn-primary'; ?>" 
                                data-user-id="<?php echo $profile_user_id; ?>">
                            <?php echo $is_following ? 'Following' : 'Follow'; ?>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="profile-tabs">
                <button class="tab-btn active" data-tab="all">All Posts</button>
                <button class="tab-btn" data-tab="images">Images</button>
                <button class="tab-btn" data-tab="videos">Videos</button>
            </div>
            
            <div class="profile-content">
                <div class="tab-content active" id="all-tab">
                    <?php if (empty($posts)): ?>
                        <div class="no-posts">
                            <p><?php echo $is_own_profile ? 'You haven\'t posted anything yet.' : 'No posts yet.'; ?></p>
                            <?php if ($is_own_profile): ?>
                                <a href="pages/create-post.php" class="btn btn-primary">Create Your First Post</a>
                            <?php endif; ?>
                        </div>
                    <?php else: ?>
                        <div class="posts-grid">
                            <?php foreach ($posts as $post): ?>
                                <div class="post-thumbnail" data-post-id="<?php echo $post['id']; ?>">
                                    <?php if ($post['media_type'] == 'image'): ?>
                                        <img src="<?php echo htmlspecialchars($post['media_url']); ?>" alt="Post">
                                    <?php else: ?>
                                        <video>
                                            <source src="<?php echo htmlspecialchars($post['media_url']); ?>" type="video/mp4">
                                        </video>
                                        <div class="video-overlay">▶</div>
                                    <?php endif; ?>
                                    <div class="post-overlay">
                                        <span class="post-stats">
                                            <span>♥ <?php echo $post['likes_count']; ?></span>
                                            <span>💬 <?php echo $post['comments_count']; ?></span>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="tab-content" id="images-tab">
                    <?php if (empty($images)): ?>
                        <div class="no-posts">
                            <p>No images posted yet.</p>
                        </div>
                    <?php else: ?>
                        <div class="posts-grid">
                            <?php foreach ($images as $post): ?>
                                <div class="post-thumbnail" data-post-id="<?php echo $post['id']; ?>">
                                    <img src="<?php echo htmlspecialchars($post['media_url']); ?>" alt="Post">
                                    <div class="post-overlay">
                                        <span class="post-stats">
                                            <span>♥ <?php echo $post['likes_count']; ?></span>
                                            <span>💬 <?php echo $post['comments_count']; ?></span>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="tab-content" id="videos-tab">
                    <?php if (empty($videos)): ?>
                        <div class="no-posts">
                            <p>No videos posted yet.</p>
                        </div>
                    <?php else: ?>
                        <div class="posts-grid">
                            <?php foreach ($videos as $post): ?>
                                <div class="post-thumbnail" data-post-id="<?php echo $post['id']; ?>">
                                    <video>
                                        <source src="<?php echo htmlspecialchars($post['media_url']); ?>" type="video/mp4">
                                    </video>
                                    <div class="video-overlay">▶</div>
                                    <div class="post-overlay">
                                        <span class="post-stats">
                                            <span>♥ <?php echo $post['likes_count']; ?></span>
                                            <span>💬 <?php echo $post['comments_count']; ?></span>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <?php include 'includes/bottom-nav.php'; ?>
    </div>
    
    <script src="assets/js/profile.js"></script>
</body>
</html>
